#ifndef __CFont__
#define __CFont__

#include <Basics/CCountedObject.hpp>
#include <Basics/CString.hpp>
#include <Collections/TAutomaticPointer.hpp>
#include "CFontReference.hpp"
#include "../Windowing/CWindowTools.hpp"

//	===========================================================================

using Exponent::Collections::TAutomaticPointer;
using Exponent::Basics::CCountedObject;
using Exponent::Basics::CString;
using Exponent::GUI::Fonts::CFontReference;
using Exponent::GUI::Windowing::CWindowTools;

//	===========================================================================

namespace Exponent
{
	namespace GUI
	{
		namespace Fonts
		{
			/**
			 * @class CFont CFont.hpp
			 * @brief Font wrapper class
			 *
			 * @date 21/09/2004
			 * @author Paul Chana
			 * @version 1.0.0 Initial version
			 *
			 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
			 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
			 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
			 * All content is the Intellectual property of Exp Digital Uk.\n
			 * Certain sections of this code may come from other sources. They are credited where applicable.\n
			 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
			 *
			 * $Id: CFont.hpp,v 1.6 2007/03/07 22:18:51 paul Exp $
			 */
			class CFont : public CCountedObject
			{
				/** @cond */
				EXPONENT_CLASS_DECLARATION;
				/** @endcond */

//	===========================================================================

			public:	

//	===========================================================================

				static CFont *CFONT_SYSTEM_FONT;						/**< The standard system font */

//	===========================================================================

				/**
				 * Construction
				 * @param fontname The natural name of the font
				 * @param height The height of the font
				 */
				CFont(const CString &fontname = "Arial", const unsigned long height = 10);

				/**
				 * Copy Construction
				 * @param font The font to copy
				 */
				CFont(const CFont &font);

				/**
				 * Destruction
				 */
				virtual ~CFont();

//	===========================================================================

				/**
				 * Assignment operator
				 * @param font The font to copy
				 */
				CFont &operator = (const CFont &font);

//	===========================================================================

				/**
				 * Set the font
				 * @param font The font to copy
				 */
				void setFont(const CFontReference &font);

				/**
				 * Set the font
				 * @param fontName The name of the font
				 * @param height The height of the font
				 * @param italic True if you want to draw in italic, false otherwise
				 * @param underline True if you want to underline the font, false otherwise
				 * @param strike True if you want to strike through the text, false otherwise
				 * @param bold True if you want a bold variant, false otherwise
				 * @note italic, underline, strike and bold are currently under development, they will work on windows, on mac they are still under construction
				 */
				void setFont(const CString &fontName, const long height, const bool italic = false, const bool underline = false, const bool strike = false, const bool bold = false);

				/**
				 * Set the height
				 * @param height The height of the font
				 */
				void setHeight(const unsigned long height);

				/**
				 * SEt the filename (loads the font)
				 * @param fontname THe name of the font
				 */
				void setFontName(const CString &fontname);

//	===========================================================================

				#ifdef WIN32
					/**
					 * Get hte handle to the font
					 * @retval HFONT The windows font handle
					 */
					HFONT getFont() const { return m_fontHandle; }
				#endif

				/**
				 * Get the font height
				 * @retval long The font height
				 */
				long getHeight() const;

				/**
				 * Get the font size
				 * @retval long The font height
				 */
				long getFontSize() const;

				/**
				 * Get the font width
				 * @retval long The font width
				 */
				long getWidth() const;

				/**
				 * Get the font name
				 * @retval const CString& The name of the font
				 */
				const CString &getFontName() const { return m_fontName; }

//	===========================================================================
				
				/**
				 * Get the length of a string in pixels
				 * @param string The string to comppute
				 * @param drawContext The drawContext, only needed for windows, on mac can be null
				 * @retval long The length of the string
				 */
				static long getStringLength(const CString &string, void *drawContext = NULL);

//	===========================================================================

				/**
				 * Create the defalt font
				 */
				static void createDefaultFont();

				/**
				 * Destroy the default font
				 */
				static void destroyDefaultFont();

				/**
				 * Compute the size of a font given the point size
				 * @param pointSize The point size of a font
				 */
				static long getFontSizeFromPointSize(const long pointSize);

//	===========================================================================

			protected:

//	===========================================================================

				/**
				 * Load the font
				 */
				void loadFont();

				/**
				 * Unload the font
				 */
				void unloadFont();

//	===========================================================================

				#ifdef WIN32
					HFONT m_fontHandle;		/**< Handle to the font */
					LOGFONT m_logFont;		/**< The log font structure */
				#endif

				CString m_fontName;			/**< What is the name of the font */
				long m_height;				/**< The actual height */
			};
		}
	}
}
#endif	// End of CFont.hpp